unit MXWriter1;

{
  Demonstrate the generation of an XML document from a database
  using the Microsoft MXWriter.
  Requires 'movie-watcher' alias to be set up in BDE.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written November 3, 2000.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, Db, DBTables, StdCtrls, ExtCtrls, CommonXML, ActiveX, MSXML2_tlb;

type
  TfrmWriterXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
  public
  end;

var
  frmWriterXML: TfrmWriterXML;

implementation

{$R *.DFM}

{ Generate the XML document as text }
procedure TfrmWriterXML.btnGenerateClick(Sender: TObject);
var
  XMLDoc: IMXWriter;
  ContentHandler: IVBSAXContentHandler;
  DTDHandler: IVBSAXDTDHandler;
  LexicalHandler: IVBSAXLexicalHandler;
  Attributes: IMXAttributes;
  Empty: WideString;
  NoValue: WideString;

  { Start a new element tag }
  procedure StartElement(Name: WideString);
  begin
    ContentHandler.startElement(
      NoValue, NoValue, Name, Attributes as IVBSAXAttributes);
    Attributes.clear;
  end;

  { End an element tag }
  procedure EndElement(Name: WideString);
  begin
    ContentHandler.endElement(NoValue, NoValue, Name);
  end;

  { Save an attribute for adding to an element }
  procedure AddAttribute(Name, Value: WideString);
  begin
    Attributes.addAttribute(NoValue, NoValue, Name, NoValue, Value);
  end;

  { Add a simple element that only contains text }
  procedure AddSimpleElement(Field: TField; AsCDATA: Boolean = False);
  var
    Value: WideString;
  begin
    StartElement(ModifyName(Field.FieldName));
    if AsCDATA then
      LexicalHandler.startCDATA;
    Value := Field.DisplayText;
    if Value = '' then
      Value := NoValue;
    ContentHandler.characters(Value);
    if AsCDATA then
      LexicalHandler.endCDATA;
    EndElement(ModifyName(Field.FieldName));
  end;

  { Include empty field tag only if flag in DB set }
  procedure AddOptElement(Field: TField);
  begin
    if Field.AsBoolean then
    begin
      StartElement(ModifyName(Field.FieldName));
      EndElement(ModifyName(Field.FieldName));
    end;
  end;

  { Compile elements for the stars of the movie }
  procedure GenerateStars;
  begin
    with datCommonXML.qryStars do
    begin
      StartElement(StarringTag);
      First;
      while not EOF do
      begin
        AddSimpleElement(FieldByName(StarField));
        Next;
      end;
      EndElement(StarringTag);
    end;
  end;

  { Generate elements for each movie }
  procedure GenerateMovies;
  var
    BaseId: string;
  begin
    StartElement(MoviesTag);
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        BaseId := FieldByName(MovieIdField).DisplayText;
        AddAttribute(Id, BaseId);
        AddAttribute(Rating, FieldByName(RatingField).DisplayText);
        if FieldByName(LogoURLField).AsString <> '' then
          AddAttribute(ModifyName(FieldByName(LogoURLField).FieldName),
            BaseId + 'Logo');
        if FieldByName(URLField).AsString <> '' then
          AddAttribute(ModifyName(FieldByName(URLField).FieldName),
            BaseId + 'Url');
        StartElement(MovieTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(LengthField));
        AddSimpleElement(FieldByName(DirectorField));
        GenerateStars;
        AddSimpleElement(FieldByName(SynopsisField), True);
        EndElement(MovieTag);
        Next;
      end;
    end;
    EndElement(MoviesTag);
  end;

  { Compile elements for the pricing schemes }
  procedure GeneratePricing;
  begin
    with datCommonXML.qryPricing do
    begin
      StartElement(PricingTag);
      First;
      while not EOF do
      begin
        AddAttribute(Id, FieldByName(PricingIdField).DisplayText);
        StartElement(PriceTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(PeriodField));
        AddSimpleElement(FieldByName(AdultField));
        AddSimpleElement(FieldByName(ChildField));
        AddSimpleElement(FieldByName(DiscountField));
        EndElement(PriceTag);
        Next;
      end;
      EndElement(PricingTag);
    end;
  end;

  { Generate elements for each cinema }
  procedure GenerateCinemas;
  begin
    StartElement(CinemasTag);
    with datCommonXML.qryCinema do
    begin
      First;
      while not EOF do
      begin
        AddAttribute(Id, FieldByName(CinemaIdField).DisplayText);
        StartElement(CinemaTag);
        AddSimpleElement(FieldByName(NameField));
        AddSimpleElement(FieldByName(PhoneField));
        AddSimpleElement(FieldByName(AddressField));
        AddSimpleElement(FieldByName(DirectionsField));
        StartElement(FacilitiesTag);
        AddOptElement(FieldByName(CandyBarField));
        AddOptElement(FieldByName(DisabledField));
        EndElement(FacilitiesTag);
        GeneratePricing;
        EndElement(CinemaTag);
        Next;
      end;
    end;
    EndElement(CinemasTag);
  end;

  { Compile elements for the sessions for each screening }
  procedure GenerateSessions;
  begin
    with datCommonXML.qrySessions do
    begin
      StartElement(SessionsTag);
      First;
      while not EOF do
      begin
        AddAttribute(PricingId, FieldByName(PricingIdField).DisplayText);
        StartElement(SessionTag);
        AddSimpleElement(FieldByName(TimeField));
        EndElement(SessionTag);
        Next;
      end;
      EndElement(SessionsTag);
    end;
  end;

  { Generate elements for each screening }
  procedure GenerateScreenings;
  begin
    StartElement(ScreeningsTag);
    with datCommonXML.qryScreening do
    begin
      First;
      while not EOF do
      begin
        AddAttribute(MovieId, FieldByName(MovieIdField).DisplayText);
        AddAttribute(CinemaId, FieldByName(CinemaIdField).DisplayText);
        StartElement(ScreeningTag);
        AddSimpleElement(FieldByName(StartDateField));
        AddSimpleElement(FieldByName(EndDateField));
        StartElement(FeaturesTag);
        AddSimpleElement(FieldByName(DigSoundField));
        EndElement(FeaturesTag);
        StartElement(RestrictionsTag);
        AddOptElement(FieldByName(NoPassesField));
        EndElement(RestrictionsTag);
        GenerateSessions;
        EndElement(ScreeningTag);
        Next;
      end;
    end;
    EndElement(ScreeningsTag);
  end;

  { Generate DTD and contents }
  procedure GenerateDTD;
  var
    Wide1, Wide2, Wide3: WideString;
    BaseId: string;
  begin
    Wide1 := MovieWatcherTag;
    Wide2 := XMLDTDFile;
    LexicalHandler.startDTD(Wide1, Empty, Wide2);
    Wide1 := JPEGType;
    Wide2 := JPEGPubId;
    Wide3 := JPEGSysId;
    DTDHandler.notationDecl(Wide1, Wide2, Wide3);
    Wide1 := HTMLType;
    Wide2 := HTMLPubId;
    Wide3 := HTMLSysId;
    DTDHandler.notationDecl(Wide1, Wide2, Wide3);
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        BaseId := FieldByName(MovieIdField).DisplayText;
        if FieldByName(LogoURLField).AsString <> '' then
        begin
          Wide1 := BaseId + 'Logo';
          Wide2 := FieldByName(LogoURLField).DisplayText;
          Wide3 := JPEGType;
          DTDHandler.unparsedEntityDecl(Wide1, Empty, Wide2, Wide3);
        end;
        if FieldByName(URLField).AsString <> '' then
        begin
          Wide1 := BaseId + 'Url';
          Wide2 := FieldByName(URLField).DisplayText;
          Wide3 := HTMLType;
          DTDHandler.unparsedEntityDecl(Wide1, Empty, Wide2, Wide3);
        end;
        Next;
      end;
    end;
    LexicalHandler.endDTD;
  end;

  { Add the document generated so far to the output }
  procedure UpdateOutput;
  begin
    memXML.Lines.Text := memXML.Lines.Text + XMLDoc.output;
    XMLDoc.output     := Empty;
  end;

  { Generate XML prolog, style sheet reference, and main element }
  procedure GenerateDocument;
  var
    Wide1, Wide2: WideString;
  begin
    ContentHandler.startDocument;
    GenerateDTD;
    Wide1 := XMLComment;
    LexicalHandler.comment(Wide1);
    Wide1 := XMLStyleTag;
    Wide2 := XMLStyleAttrs;
    ContentHandler.processingInstruction(Wide1, Wide2);
    UpdateOutput;
    StartElement(MovieWatcherTag);
    GenerateMovies;
    UpdateOutput;
    GenerateCinemas;
    UpdateOutput;
    GenerateScreenings;
    EndElement(MovieWatcherTag);
    ContentHandler.endDocument;
    UpdateOutput;
  end;

begin
  Screen.Cursor       := crHourGlass;
  btnGenerate.Enabled := False;
  try
    Empty   := '';
    NoValue := ' ';
    memXML.Lines.Clear;
    { Instantiate the XML writer }
    XMLDoc         := CoMXXMLWriter.Create;
    ContentHandler := XMLDoc as IVBSAXContentHandler;
    DTDHandler     := XMLDoc as IVBSAXDTDHandler;
    LexicalHandler := XMLDoc as IVBSAXLexicalHandler;
    Attributes     := CoSAXAttributes.Create;
    XMLDoc.indent  := True;
    { Generate the structure }
    GenerateDocument;
    { Release the XML writer }
    Attributes := nil;
    XMLDoc     := nil;
  finally
    btnGenerate.Enabled := True;
    Screen.Cursor       := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmWriterXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

initialization
  CoInitialize(nil);
finalization
  CoUninitialize;
end.
